﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;
using TMPro;
using UnityEngine.EventSystems;
using System;

[System.Serializable]
public class TextItemsSelector : Selectable, IPointerEnterHandler, IPointerExitHandler
{
    //Event in editor to allow code to run when selected
    [SerializeField]
    private Button.ButtonClickedEvent _OnClick = new Button.ButtonClickedEvent();

    public Button.ButtonClickedEvent onClick
    {
        get
        {
            return _OnClick;
        }

        set
        {
            _OnClick = value;
        }
    }

    #region EventArgs Classes
    public class SelectedItemChangedEventArgs : EventArgs
    {
        public string OldItemText { get; set; }
        public string NewItemText { get; set; }
        public int OldIndex { get; set; }
        public int NewIndex { get; set; }
    }

    public event EventHandler<SelectedItemChangedEventArgs> SelectedItemChanged;    //Called when the selected item changes, sends custom event args
    #endregion

    //Public properties
    [SerializeField]
    public string[] Items;  //The text items to display
    public int DefaultIndex = 0;    //Allows changing the default displayed text
    public bool ShouldWrap = true;

    //UI controls
    public TMP_Text Text;
    public Button LeftArrow;
    public Button RightArrow;

    //Animations
    private Animator _Animator;
    private Animator _LeftArrowAnimator;
    private Animator _RightArrowAnimator;

    public int CurrentIndex { get; private set; }   //The index of the currently visible text
    public string SelectedItemText { get; private set; }    //The currently visible text

    protected override void Awake()
    {
        base.Awake();
        CurrentIndex = DefaultIndex;

        //Get components, subscribe to events
        LeftArrow.onClick.AddListener(OnLeftArrowClick);
        RightArrow.onClick.AddListener(OnRightArrowClick);
        _Animator = GetComponent<Animator>();
        _LeftArrowAnimator = LeftArrow.GetComponent<Animator>();
        _RightArrowAnimator = RightArrow.GetComponent<Animator>();

        //Sanity checks, then display the desired text
        if (CurrentIndex >= 0 && CurrentIndex < Items.Length)
        {
            SelectedItemText = Items[CurrentIndex];
            Text.SetText(SelectedItemText);
        }
    }

    /// <summary>
    /// Handler for the left arrow being clicked, updates the current index and raises the event
    /// </summary>
    private void OnLeftArrowClick()
    {
        int oldIndex = 0;

        //Play the animation, decrement the indexes, prep the event args
        if (CurrentIndex - 1 < 0)
        {
            if (ShouldWrap)
            {
                oldIndex = 0;
                CurrentIndex = Items.Length;
            }

            else
            {
                return;
            }
        }

        else
        {
            oldIndex = CurrentIndex;
        }

        CurrentIndex--;

        //Update the UI, play the audio
        SelectedItemText = Items[CurrentIndex];
        Text.SetText(SelectedItemText);
        AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles[Constants.UIButtonSelectAudioID]);

        //Setup the event args and send the event notification to all subscribers
        SelectedItemChangedEventArgs args = new SelectedItemChangedEventArgs();

        args.OldItemText = Items[oldIndex];
        args.NewItemText = SelectedItemText;
        args.OldIndex = oldIndex;
        args.NewIndex = CurrentIndex;

        OnSelectedItemChanged(args);
    }

    /// <summary>
    /// Handler for the right arrow being clicked, updates the current index and raises the event
    /// </summary>
    private void OnRightArrowClick()
    {
        int oldIndex = 0;

        //Play the animation, increment the indexes, prep the event args
        if (CurrentIndex + 1 == Items.Length)
        {
            if (ShouldWrap)
            {
                oldIndex = Items.Length - 1;
                CurrentIndex = -1;
            }

            else
            {
                return;
            }
        }

        else
        {
            oldIndex = CurrentIndex;
        }

        CurrentIndex++;

        //Update the UI, play the audio
        SelectedItemText = Items[CurrentIndex];
        Text.SetText(SelectedItemText);
        AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles[Constants.UIButtonSelectAudioID]);


        //Setup the event args and send the event notification to all subscribers
        SelectedItemChangedEventArgs args = new SelectedItemChangedEventArgs();

        args.OldItemText = Items[oldIndex];
        args.NewItemText = SelectedItemText;
        args.OldIndex = oldIndex;
        args.NewIndex = CurrentIndex;

        OnSelectedItemChanged(args);
    }

    private void OnSelectedItemChanged(SelectedItemChangedEventArgs e)
    {
        //Called when the selected item changes and sends an event notification to all subscribers
        EventHandler<SelectedItemChangedEventArgs> handler = SelectedItemChanged;
        if (handler != null)
        {
            handler(this, e);
        }
    }
}
